#include <formatio.h>
#include <ansi_c.h>
#include <visa.h>		  
#include "ax1360P.h"

/*==================================================================================*/
#define AX1360P_REVISION   "Rev 1.0, 06/2011,CVI 9.0.1"/* Instrument driver revision*/
#define BUFFER_SIZE        512L                      /* File I/O buffer size        */
#define INVALID_SAMPLE     0xFFFF                    /* Sample validity criterion   */
#define MAX_MEM_SIZE       100300                    /* Memory needed for file I/O  */
#define TMO_VALUE          1000                      /* Time out value              */
/*==================================================================================*/

/*=AX1360P===================================================================*/
/* LabWindows/CVI 9.0.1 Instrument Driver (VISA I/O)                         */
/* Original Release: June 2011                                               */
/*                                                                           */                           
/* Metrix Technical Support: Olivier Lemaire                                 */
/*                                                                           */
/* For LabWindows Technical Support in the United States call:               */
/*              National Instruments, Austin, Texas                          */
/*              Ph (800) 433-3488    Fax (512) 794-5678                      */
/*                                                                           */
/*                                                                           */
/* Modification History: None                                                */
/*                                                                           */
/*===========================================================================*/
 
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS =====================================*/
/*****************************************************************************/

static ViString cmdVertCoupling[] = {"AC","DC","GRO"};       
static ViString cmdcalcfft[] = {"RECT", "HAMM", "HANN", "BLAC"};

                                     
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/*  ax1360P_stringValPair is used in the  ax1360P_errorMessage function      */
/*===========================================================================*/
typedef struct   ax1360P_stringValPair
{
   ViStatus stringVal;
   ViPString stringName;
}   ax1360P_tStringValPair;
	 
struct ax1360P_statusDataRanges {
    ViInt16 triggerMode;
    ViInt16 val2;
    ViInt16 val3;
    ViChar instrDriverRevision[256];
    ViInt16 instrMode;
};
typedef struct ax1360P_statusDataRanges * ax1360P_instrRange;


/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean ax1360P_invalidViBooleanRange (ViBoolean val);
ViBoolean ax1360P_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean ax1360P_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean ax1360P_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean ax1360P_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean ax1360P_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean ax1360P_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus  ax1360P_initCleanUp (ViSession openRMSession, ViPSession openInstrSession, ViStatus currentStatus);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function:	Initialize                                                   */
/* Purpose:		This function opens the instrument, queries the instrument   */
/*				for its ID, and initializes the instrument to a known state  */
/*===========================================================================*/
ViStatus _VI_FUNC  ax1360P_init (ViRsrc resourceName, ViBoolean IDQuery,
					ViPSession instrSession)
{
	ViStatus  ax1360P_status = VI_SUCCESS;
	ViSession rmSession = 0;
	ViUInt32 retCnt = 0;
	ViByte rdBuffer[BUFFER_SIZE];
	ViByte wrBuffer[BUFFER_SIZE];
    
	/*- Check input parameter ranges ----------------------------------------*/
	if ( ax1360P_invalidViBooleanRange (IDQuery))
		return VI_ERROR_PARAMETER2;

	/*- Open instrument session ---------------------------------------------*/
	if (( ax1360P_status = viOpenDefaultRM (&rmSession)) < 0)
		return  ax1360P_status;

	if (( ax1360P_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0) 
		{
		viClose (rmSession);
		return  ax1360P_status;
		}	

	/*- Configure VISA Formatted I/O ----------------------------------------*/
	if (( ax1360P_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_BAUD, 9600)) < 0)
		return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);
	if (( ax1360P_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_FLOW_CNTRL, VI_ASRL_FLOW_NONE)) < 0)
		return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);
	if (( ax1360P_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_DATA_BITS, 8)) < 0)
		return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);
	if (( ax1360P_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_PARITY, VI_ASRL_PAR_NONE)) < 0)
		return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);
	if (( ax1360P_status = viSetAttribute (*instrSession, VI_ATTR_ASRL_STOP_BITS, VI_ASRL_STOP_ONE)) < 0)
		return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);
	if (( ax1360P_status = viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, TMO_VALUE)) < 0)
		return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);
	if (( ax1360P_status = viSetAttribute (*instrSession, VI_ATTR_TERMCHAR, 0x0D)) < 0)
		return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);
	if (( ax1360P_status = viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 512)) < 0)
		return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);
	if (( ax1360P_status = viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE, VI_FLUSH_ON_ACCESS)) < 0)
		return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);
	if (( ax1360P_status = viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE, VI_FLUSH_ON_ACCESS)) < 0)
		return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);

	viWrite (*instrSession, "\n", 1, &retCnt);
	
	/*- Identification Query ------------------------------------------------*/
	if (IDQuery) 
		{
		Fmt (wrBuffer, "*IDN?\r\n");
		if ((ax1360P_status = viWrite (*instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return  ax1360P_initCleanUp (rmSession, instrSession,  ax1360P_status);
		if ((ax1360P_status = viRead (*instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
			return  ax1360P_status;
		if (FindPattern (rdBuffer, 0, -1, "SN:", 1, 0) == -1)
			return  VI_ERROR_FAIL_ID_QUERY;
		}

	return  ax1360P_status = VI_SUCCESS;
}

/*===========================================================================*/
/* Function: Configure AX1360P Voltage Settings                              */
/* Purpose:  This function configures the voltage settings for a specified   */
/*           channel                                                         */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_SetVoltage (ViSession instrSession, ViInt16 input, ViReal64 voltage)
{
	ViStatus ax1360P_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (ax1360P_invalidViInt16Range(input, 1, 2))
		return VI_ERROR_PARAMETER2;
	if (ax1360P_invalidViReal64Range(voltage, 0, 32))
		return VI_ERROR_PARAMETER3;
    						  
	/*-Send vertical config command------------------------------------------*/
	Fmt (wrBuffer, "VSET%i:%f[p1]\r\n", input, voltage);
	if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return ax1360P_status; 
        
	return ax1360P_status;
}

/*===========================================================================*/
/* Function: Configure AX1360P Current Settings                              */
/* Purpose:  This function configures the current settings for a specified   */
/*           channel                                                         */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_SetCurrent (ViSession instrSession, ViInt16 input, ViReal64 current)
{
	ViStatus ax1360P_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (ax1360P_invalidViInt16Range(input, 1, 2))
		return VI_ERROR_PARAMETER2;
	if (ax1360P_invalidViReal64Range(current, 0, 3.2))
		return VI_ERROR_PARAMETER3;
    						  
	/*-Send vertical config command------------------------------------------*/
	Fmt (wrBuffer, "ISET%i:%f[p2]\r\n", input, current);
	if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return ax1360P_status; 
        
	return ax1360P_status;
}

/*===========================================================================*/
/* Function: Configure AX1360P Tracking Settings                             */
/* Purpose:  This function configures the tracking settings                  */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_SetTracking (ViSession instrSession, ViInt16 tracking)
{
	ViStatus ax1360P_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (ax1360P_invalidViInt16Range(tracking, 0, 2))
		return VI_ERROR_PARAMETER2;
    						  
	/*-Send vertical config command------------------------------------------*/
	Fmt (wrBuffer, "TRACK%i\r\n", tracking);
	if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return ax1360P_status; 
        
	return ax1360P_status;
}

/*===========================================================================*/
/* Function: Configure AX1360P Outpout                                       */
/* Purpose:  This function configures the outpout settings                   */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_Output (ViSession instrSession, ViBoolean outpout)
{
	ViStatus ax1360P_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Check input parameter ranges------------------------------------------*/
	if (ax1360P_invalidViBooleanRange(outpout))
		return VI_ERROR_PARAMETER2;
    						  
	/*-Send vertical config command------------------------------------------*/
	Fmt (wrBuffer, "OUT%i\r\n", outpout);
	if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return ax1360P_status; 
        
	return ax1360P_status;
}

/*===========================================================================*/
/* Function: Read V out                                                      */
/* Purpose:  This function reads V out.                                      */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_readVout (ViSession instrSession, ViPReal64 v1, ViPReal64 v2)
{                              
	ViStatus ax1360P_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Send CH1 Vout command-------------------------------------------------*/
	Fmt (wrBuffer, "VOUT1?\r\n");
	if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return ax1360P_status; 
	if ((ax1360P_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return  ax1360P_status;
	Scan (rdBuffer, "%f", v1);

	/*-Send CH1 Vout command-------------------------------------------------*/
	Fmt (wrBuffer, "VOUT2?\r\n");
	if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return ax1360P_status; 
	if ((ax1360P_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return  ax1360P_status;
	Scan (rdBuffer, "%f", v2);

	return ax1360P_status;
}

/*===========================================================================*/
/* Function: Read I out                                                      */
/* Purpose:  This function reads I out.                                      */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_readIout (ViSession instrSession, ViPReal64 i1, ViPReal64 i2)
{                              
	ViStatus ax1360P_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViByte rdBuffer[BUFFER_SIZE];
	ViInt32 retCnt;

	/*-Send CH1 Vout command-------------------------------------------------*/
	Fmt (wrBuffer, "IOUT1?\r\n");
	if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return ax1360P_status; 
	if ((ax1360P_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return  ax1360P_status;
	Scan (rdBuffer, "%f", i1);

	/*-Send CH1 Vout command-------------------------------------------------*/
	Fmt (wrBuffer, "IOUT2?\r\n");
	if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
		return ax1360P_status; 
	if ((ax1360P_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return  ax1360P_status;
	Scan (rdBuffer, "%f", i2);

	return ax1360P_status;
}

/*===========================================================================*/
/* Function: Save/Recall Setup From/To Instrument                            */
/* Purpose:  This function allows you to save the current setup of the       */
/*           instrument or to recall a setup stored.                         */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_saveRecall (ViSession instrSession, ViBoolean mode, ViInt16 memory)
{
	ViStatus ax1360P_status = VI_SUCCESS;
	ViUInt32 retCnt;
	ViByte wrBuffer[BUFFER_SIZE];

	/*-Check input parameter ranges------------------------------------------*/
	if (ax1360P_invalidViBooleanRange(mode))
		return VI_ERROR_PARAMETER2;
	if (ax1360P_invalidViInt16Range(memory, 1, 4))
		return VI_ERROR_PARAMETER3;

	/*-Save setup from AX1360P------------------------------------------------*/
	if (mode) 	
		{
		Fmt (wrBuffer, "SAV%i\r\n", memory);
		if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return ax1360P_status; 
		}
	/*-Recall setup to AX1360P------------------------------------------------*/
	else 
		{
		Fmt (wrBuffer, "RCL%i\r\n", memory);
		if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen (wrBuffer), &retCnt)) < 0)
			return ax1360P_status; 
		}
    
	 
	return ax1360P_status;
}

/*===========================================================================*/
/* Function: Write To Instrument                                             */
/* Purpose:  This function writes a command string to the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_writeInstrData (ViSession instrSession, ViString writeBuffer)
{
	ViStatus ax1360P_status = VI_SUCCESS;

	if ((ax1360P_status = viPrintf (instrSession, "%s", writeBuffer)) < 0)
		return ax1360P_status;

	return ax1360P_status;
}

/*===========================================================================*/
/* Function: Read Instrument Buffer                                          */
/* Purpose:  This function reads the output buffer of the instrument.        */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_readInstrData (ViSession instrSession, ViInt32 numBytes,
							ViChar _VI_FAR rdBuf[], ViPInt32 bytesRead)
{
	ViStatus ax1360P_status = VI_SUCCESS;
	*bytesRead = 0L;

	if ((ax1360P_status = viRead (instrSession, rdBuf, BUFFER_SIZE, bytesRead)) < 0)
		return ax1360P_status;

	return ax1360P_status;
}


/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_errorMessage (ViSession instrSession, ViStatus errorCode,
							ViChar _VI_FAR errMessage[])
{
	ViStatus ax1360P_status = VI_SUCCESS;
	ViInt16 i;
    
	static ax1360P_tStringValPair statusDescArray[] = {
		{VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
		{VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
		{VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
		{VI_ERROR_PARAMETER4,   "ERROR: Parameter 4 out of range"},
		{VI_ERROR_PARAMETER5,   "ERROR: Parameter 5 out of range"},
		{VI_ERROR_PARAMETER6,   "ERROR: Parameter 6 out of range"},
		{VI_ERROR_PARAMETER7,   "ERROR: Parameter 7 out of range"},
		{VI_ERROR_PARAMETER8,   "ERROR: Parameter 8 out of range"},
		{VI_ERROR_INSTR_PARAMETER9,   "ERROR: Parameter 9 out of range"},
		{VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
		{VI_ERROR_INSTR_FILE_OPEN,    "ERROR: Opening the specified file"},
		{VI_ERROR_INSTR_FILE_WRITE,   "ERROR: Writing to the specified file"},
		{VI_ERROR_INSTR_INTERPRETING_RESPONSE,           "ERROR: Interpreting the instrument's response"},
		{VI_ERROR_AX1360P_ERROR,                  "ERROR: Instrument specific error"},
		{VI_ERROR_AX1360P_INCONSISTENT_PARAMETER3,"ERROR: Parameter 3 is INCONSISTENT"},
		{VI_ERROR_AX1360P_INCONSISTENT_PARAMETER5,"ERROR: Parameter 5 is INCONSISTENT"},
		{VI_ERROR_AX1360P_INCONSISTENT_PARAMETER6,"ERROR: Parameter 6 is INCONSISTENT"},
		{VI_ERROR_AX1360P_INCONSISTENT_PARAMETER7,"ERROR: Parameter 7 is INCONSISTENT"},
		{VI_ERROR_AX1360P_AUTOTEST_PROBLEM,       "ERROR: Performing a self-test"}, 
		{VI_ERROR_AX1360P_CHAN_NOT_ACTIVE,        "ERROR: Channel is not active"},
		{VI_ERROR_AX1360P_NO_CHAN_ENABLED,        "ERROR: No channel is enabled"},
		{VI_ERROR_AX1360P_FILE_NOT_FOUND,         "ERROR: File not available on support"},
		{VI_ERROR_AX1360P_EXISTING_FILE,          "ERROR: Filename already used"},
		{VI_ERROR_AX1360P_INVALID_FILE_CONTENT,   "ERROR: File content is invalid"},
		{VI_NULL, VI_NULL}  };
	          
	ax1360P_status = viStatusDesc (instrSession, errorCode, errMessage);
	if (ax1360P_status == VI_WARN_UNKNOWN_STATUS) 
		{
		for (i=0; statusDescArray[i].stringName; i++) 
			{
			if (statusDescArray[i].stringVal == errorCode) 
				{
				Fmt (errMessage, "%s<%s", statusDescArray[i].stringName);
				return (VI_SUCCESS);
				}
			}
		Fmt (errMessage, "%s<Unknown Error 0x%x[uw8p0]", errorCode);
		return (VI_WARN_UNKNOWN_STATUS);
		}
	ax1360P_status = VI_SUCCESS;
	return ax1360P_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*===========================================================================*/
ViStatus _VI_FUNC ax1360P_revisionQuery (ViSession instrSession,
					ViChar _VI_FAR driverRev[], ViChar _VI_FAR instrRev[])
{
	ViUInt32 retCnt = 0;
	ViStatus ax1360P_status = VI_SUCCESS;
	ViByte wrBuffer[BUFFER_SIZE];
	ViByte rdBuffer[BUFFER_SIZE];

	Fmt (wrBuffer, "*IDN?\r\n");
	if ((ax1360P_status = viWrite (instrSession, wrBuffer, strlen(wrBuffer), &retCnt)) < 0)
		return ax1360P_status;
	if ((ax1360P_status = viRead (instrSession, rdBuffer, BUFFER_SIZE, &retCnt)) < 0)
		return ax1360P_status;
	rdBuffer[retCnt] = 0;
	if (Scan (rdBuffer, "%s[t10]", instrRev) != 1)
		return VI_ERROR_INSTR_INTERPRETING_RESPONSE;
	Fmt (driverRev, "%s<%s", AX1360P_REVISION);

	return ax1360P_status;
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC  ax1360P_close (ViSession instrSession)
{

	ax1360P_instrRange instrPtr;
	ViSession rmSession;
	ViStatus  ax1360P_status = VI_SUCCESS;

	if (( ax1360P_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
		return  ax1360P_status;
	if (( ax1360P_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
		return  ax1360P_status;
    
	if(instrPtr != NULL)
		free (instrPtr);
    
	ax1360P_status = viClose (instrSession);
	viClose (rmSession);

	return  ax1360P_status;
}

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean  ax1360P_invalidViBooleanRange (ViBoolean val)
{
	return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  ax1360P_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  ax1360P_invalidViInt32Range  (ViInt32 val, ViInt32 min, ViInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  ax1360P_invalidViUInt16Range  (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range,  the return value is VI_TRUE, otherwise the return    */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean  ax1360P_invalidViUInt32Range  (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  ax1360P_invalidViReal32Range  (ViReal32 val, ViReal32 min, ViReal32 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean  ax1360P_invalidViReal64Range  (ViReal64 val, ViReal64 min, ViReal64 max)
{
	return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the  ax1360P_init function.  When */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from  ax1360P_init to     */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus  ax1360P_initCleanUp (ViSession openRMSession,ViPSession openInstrSession,
                    ViStatus currentStatus)
{
	viClose (*openInstrSession);
	viClose (openRMSession);
	*openInstrSession = VI_NULL;
    
	return currentStatus;
}


